import os
import sys
import subprocess
import platform
import shutil
from datetime import datetime

# ===================== CONFIG ===================== #

TEMP_VENV = ".anthro_tmp_env"
REQUIRED_PACKAGES = ["pillow", "mutagen"] # Added mutagen for audio tags

FONT_PATH = "Orbitron-VariableFont_wght.ttf"
BRANDING_IMAGE = "branding.png"
OUTPUT_DIR = "output"

BACKGROUND_COLOR = (0, 0, 0)
TEXT_COLOR = (255, 255, 255)

BRAND_WIDTH_RATIO = 0.35
LINE_SPACING = 1.4 
BUFFER_RATIO = 0.02 

# ================================================== #
# ---------------- VENV BOOTSTRAP ------------------ #

def is_venv():
    return sys.prefix != sys.base_prefix

def venv_python():
    if platform.system() == "Windows":
        return os.path.join(TEMP_VENV, "Scripts", "python.exe")
    return os.path.join(TEMP_VENV, "bin", "python")

def ensure_env():
    if not os.path.exists(TEMP_VENV):
        print("Creating environment and installing dependencies (Pillow, Mutagen)...")
        subprocess.check_call([sys.executable, "-m", "venv", TEMP_VENV])
        py = venv_python()
        subprocess.check_call([py, "-m", "pip", "install", "--upgrade", "pip"])
        subprocess.check_call([py, "-m", "pip", "install", *REQUIRED_PACKAGES])

# ---------------- METADATA LOGIC ------------------ #

def get_wav_info(wav_path):
    """Extracts Title, Artist, and Duration from WAV file."""
    from mutagen.wave import WAVE
    from mutagen.id3 import ID3
    
    info = {"title": os.path.basename(wav_path), "artist": "Unknown Artist", "duration": "0:00"}
    
    try:
        audio = WAVE(wav_path)
        # Handle duration
        mins, secs = divmod(int(audio.info.length), 60)
        info["duration"] = f"{mins}:{secs:02d}"
        
        # Try to get ID3 tags if they exist in the WAV
        try:
            tags = ID3(wav_path)
            info["title"] = tags.get("TIT2", info["title"])
            info["artist"] = tags.get("TPE1", info["artist"])
        except:
            # Fallback to RIFF tags if ID3 fails
            info["title"] = audio.get("TITL", [info["title"]])[0]
            info["artist"] = audio.get("TART", [info["artist"]])[0]
    except Exception as e:
        print(f"  ! Could not read metadata for {wav_path}: {e}")
        
    return info

def load_template_lines(txt_path):
    """Loads the global metadata.txt file."""
    timestamp = datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S UTC")
    lines = []
    if not os.path.exists(txt_path):
        return ["[METADATA.TXT NOT FOUND]"]

    with open(txt_path, "r", encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line: continue
            if "[TIMESTAMP]" in line:
                line = line.replace("[TIMESTAMP]", timestamp)
            lines.append(line)
    return lines

# ---------------- FONT FITTING -------------------- #

def compute_best_font(draw, font_path, lines, max_width):
    from PIL import ImageFont
    low, high = 10, max_width // 4
    best_size = low
    target_width = max_width * 0.90 

    while low <= high:
        mid = (low + high) // 2
        try:
            font = ImageFont.truetype(font_path, mid)
            widest = max(draw.textlength(str(line), font=font) for line in lines)
            if widest <= target_width:
                best_size = mid
                low = mid + 1
            else:
                high = mid - 1
        except:
            high = mid - 1
    return ImageFont.truetype(font_path, best_size)

# ---------------- IMAGE PROCESSING ---------------- #

def stamp_images(metadata_file):
    from PIL import Image, ImageDraw

    if not os.path.exists(FONT_PATH):
        raise FileNotFoundError(f"Font missing: {FONT_PATH}")
    if not os.path.exists(BRANDING_IMAGE):
        raise FileNotFoundError(f"Branding image missing: {BRANDING_IMAGE}")

    global_lines = load_template_lines(metadata_file)
    os.makedirs(OUTPUT_DIR, exist_ok=True)
    branding = Image.open(BRANDING_IMAGE).convert("RGBA")

    for filename in os.listdir("."):
        base, ext = os.path.splitext(filename)
        if ext.lower() in SUPPORTED_EXTENSIONS:
            dst = os.path.join(OUTPUT_DIR, filename)
            if os.path.exists(dst):
                continue

            # Check for matching WAV
            wav_path = base + ".wav"
            current_stamp_lines = []
            
            if os.path.exists(wav_path):
                print(f"🎵 Found Audio: {wav_path}")
                audio_info = get_wav_info(wav_path)
                current_stamp_lines.append(f"TRACK: {audio_info['title']}")
                current_stamp_lines.append(f"ARTIST: {audio_info['artist']}")
                current_stamp_lines.append(f"LENGTH: {audio_info['duration']}")
                current_stamp_lines.append("-" * 20) # Separator
            
            current_stamp_lines.extend(global_lines)

            with Image.open(filename) as img:
                img = img.convert("RGBA")
                w, h = img.size
                
                draw_tmp = ImageDraw.Draw(img)
                font = compute_best_font(draw_tmp, FONT_PATH, current_stamp_lines, w)
                
                ascent, descent = font.getmetrics()
                line_step = int((ascent + descent) * LINE_SPACING)
                total_text_h = len(current_stamp_lines) * line_step

                brand_w = int(w * BRAND_WIDTH_RATIO)
                brand_h = int(branding.height * (brand_w / branding.width))
                branding_resized = branding.resize((brand_w, brand_h), Image.LANCZOS)

                pad = max(30, int(h * BUFFER_RATIO))
                extra_h = (pad * 3) + brand_h + total_text_h
                
                canvas = Image.new("RGBA", (w, h + extra_h), BACKGROUND_COLOR)
                canvas.paste(img, (0, 0))
                draw = ImageDraw.Draw(canvas)

                # Draw Branding
                y_cursor = h + pad
                canvas.paste(branding_resized, ((w - brand_w) // 2, y_cursor), branding_resized)

                # Draw Text
                y_cursor += brand_h + pad
                for line in current_stamp_lines:
                    left, top, right, bottom = draw.textbbox((0, 0), line, font=font)
                    draw.text(((w - (right - left)) // 2, y_cursor), line, fill=TEXT_COLOR, font=font)
                    y_cursor += line_step

                canvas.convert("RGB").save(dst, quality=95, subsampling=0)
                print(f"✔ Processed: {filename}")

SUPPORTED_EXTENSIONS = (".png", ".jpg", ".jpeg", ".webp", ".tiff", ".bmp")

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python STAMP_IMAGES.py metadata.txt")
        sys.exit(1)

    if not is_venv():
        ensure_env()
        py = venv_python()
        subprocess.check_call([py, os.path.abspath(__file__), sys.argv[1]])
        sys.exit(0)

    stamp_images(sys.argv[1])